<?php

namespace App\Helpers\Filters;

use App\Helpers\Filters;

/**
 * Filter
 *
 */
class Attribute
{

    public $key;
    public $id = null; // used for dynamic filters
    public $key_column = null; // used for dynamic filters
    public $values = [];
    public $values_with_slugs = [];
    public $name;
    public $is_external;
    public $table;
    public $column;
    public $scope;
    public $type = 'in';
    public $linked_with;
    public $use_slug = false;
    public $one_value_if_filter; // if another filter is set, only one value possible.
    public $constrains = [];

    public function __construct(Filters &$parent, $key, $options = [])
    {
        $this->parent = $parent;
        $this->key = $key;
        // add also options..
        foreach ($options as $key => $value) {
            $this->$key = $value;
        }
    }

    private function getUniqueKey($key)
    {
        return $key;
    }

    public function shouldAllowOnlyOneValue()
    {
        return $this->one_value_if_filter && $this->parent->hasAttribute($this->one_value_if_filter);
    }

    private function processValue($value)
    {
        return is_object($value) ? $value->{$this->column} : $value;
    }

    private function exists($key)
    {
        return isset($this->values[$this->getUniqueKey($key)]);
    }

    public function has($value)
    {
        // this will always override
//        if ($this->shouldAllowOnlyOneValue()) {
//            return false;
//        }

        $value = $this->processValue($value);

        $has = in_array($value, $this->values);

        if ($has) {
            return true;
        }

        // in case of ranges, we have to use a different approach!
        if ($this->type === 'range' && $value >= $this->getFirst() && $value <= $this->getLast()) {
            return true;
        }
        return false;
    }

    public function add($value, $slug = null)
    {

        if ($this->shouldAllowOnlyOneValue()) {
            $this->values = [];
            $this->values_with_slugs = [];
        }

        // sometimes we may want to add slug from the object
        $slug = $this->use_slug && is_object($value) && !$slug ? $value->slug : $slug;

        // sometimes we need to show a filter value, and with - doesnt look to firendly.
        $text = str_replace(Filters::SEPARATOR_SLUG, ' ', $slug);

        // we may get a Model object.
        $value = $this->processValue($value);


        // process value, slug, text
        foreach ((array)$value as $value) {
            if (!$this->exists($value)) {
                $this->values[$value] = is_numeric($value) ? (int)$value : $value;
                $this->values_with_slugs[$value] = compact('value', 'slug', 'text');
            }
        }
    }

    public function delete($value = null)
    {
        $value = $this->processValue($value);

        if (!$value) {
            $this->parent->deleteAttribute($this->key);
        }

        unset($this->values[$value]);
        unset($this->values_with_slugs[$value]);
    }

    public function getFirst()
    {
        return reset($this->values);
    }

    public function getLast()
    {
        return end($this->values);
    }

    public function getValues()
    {
        return $this->values;
    }

    public function hasRange()
    {
        return count($this->values) > 1 && array_values($this->values) == range(reset($this->values), reset($this->values) + count($this->values) - 1);
    }

    public function sort()
    {
        asort($this->values);
        ksort($this->values_with_slugs);
    }

    /**
     * Return an array like [key, val1, val2, val3-slug, val4]
     * @return type
     */
    public function toArray()
    {
        $key = $this->id ? implode($this->parent::SEPARATOR_SLUG, [$this->id, $this->key]) : $this->key;
        return [$key] + array_map(function ($value) {
                // return value combined with slug, but without text.
                return $value['slug'] ? implode($this->parent::SEPARATOR_SLUG, array_diff_key($value, ['text' => 0])) : $value['value'];
            }, $this->values_with_slugs);
    }

    public function toString()
    {
        return implode($this->type == 'range' ? ' - ' : ', ', array_map(function ($value) {
            return !empty($value['text']) ? $value['text'] : $value['value'];
        }, $this->values_with_slugs));
    }

    /**
     * Get all possible values, if it's a range return all elements.
     * @return type
     */
    public function getPossibleValues()
    {
        if ($this->type === 'range' && is_int($this->getLast()) && $this->getLast() - $this->getFirst() <= 100) {
            $range = range($this->getFirst(), $this->getLast());
            return array_combine($range, $range);
        }
        return $this->values;
    }

    /**
     * Sometimes we need to return all the possible values except a value.
     * (eg: in case we need to toggle something in marime_intre)
     *
     * @param type $value
     * @return type
     */
    public function getDiffValues($value)
    {
        return array_diff($this->getPossibleValues(), [$this->processValue($value)]);
    }

    /**
     *  Prepare Eloquent query constrains.
     *
     * @param type $query
     * @return type
     */
    public function getConstrains($query)
    {
        foreach ($this->constrains as $constrain) {
            //list($col, $op, $val) = $constrain;
            $query->where(...$constrain);
        }

        return $query;
    }

    /**
     * Build query for da attribute.
     * @param type $baseModel
     * @return type
     */
    public function buildQuery($baseModel)
    {
        $attribute = &$this;

        if ($this->is_external) {
            $baseModel->whereHas($this->table, function ($query) use ($attribute) {
                if ($this->id && $this->key_column) {
                    $query->where($this->key_column, $this->id);
                }
                if ($attribute->type === 'range') {
                    // if range is the same number, a error will trigger.
                    $query->whereBetween($attribute->column, array_pad($attribute->values, 2, array_values($attribute->values)[0]));
                } else {
                    $query->whereIn($attribute->column, $attribute->values);
                }
                $query = $attribute->getConstrains($query);
            });
        } else {

            switch ($this->type) {
                case 'in':
                    $baseModel->whereIn($this->column, $this->values);
                    break;
                case 'range';
                    $baseModel->whereBetween($this->column, $this->values);
                    break;
                case 'gt';
                    $baseModel->where($this->column, '>=', (int)$this->values);
                    break;
                case 'scope';
                    $baseModel->{$this->scope}();
                    break;
                case 'order_desc';
                    $baseModel->orderByDesc($this->column);
                    break;
                case 'dummy':
                    break;
            }
            $baseModel = $this->getConstrains($baseModel);
        }
        return $baseModel;
    }

}
