<?php

namespace App\Model;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Utils;
use Str;

class Brand extends Model
{

    use TranslatableColumn;
    use Filterable;

    protected $guarded = ['id'];
    protected $filter_name = 'brand';

    public static function boot()
    {
        parent::boot();

        static::saving(function ($model) {
            $model->slug = Str::slug($model->name);
        });

        // we want the categories to be sorted always by priority.
        static::addGlobalScope('order', function (Builder $builder) {
            $builder->orderBy('slug', 'asc');
        });
    }

    public function scopeOrdered($query)
    {
        $query->orderBy('slug');
    }

    public function scopePriority($query)
    {
        $query->orderBy('priority');
    }

    public function scopeFeatured($query)
    {
        $query->where('is_featured', true);
    }


    public function scopeWines($query)
    {

        $query->whereHas('products.category', function ($q) {
            $q->where('id', config('categories_mapping.wines.id'));
        });
    }

    public function products()
    {
        return $this->hasMany('App\Model\Product');
    }

    public function photo()
    {
        return $this->belongsTo('App\Model\Brand\BrandPhoto');
    }


//    public function categories()
//    {
//        return $this->hasManyThrough('App\Model\CategoryProduct', 'App\Model\Product');
//    }

//    public function category()
//    {
//        return $this->hasManyThrough('App\Model\Category', 'App\Model\Product', 'category_id', 'id');
//    }

    public function getBrandIdAttribute($value)
    {
        return $this->id;
    }

    public function getSlugAttribute($value)
    {

        return $value ?: Str::slug($this->name);
    }

    public function getLogoUrl()
    {
        if (!$this->logo) {
            return $this->getDummyImage();
        }
        return Utils::storageUrl('logo', $this->logo);
    }

//    public function getPhotoUrl()
//    {
//        return Utils::storageUrl('brand_photos', $this->photo);
//    }

    public function getDummyImage()
    {
        return 'data:image/gif;base64,R0lGODlhAQABAIAAAMLCwgAAACH5BAAAAAAALAAAAAABAAEAAAICRAEAOw==';
    }

    public function getPhotoUrl()
    {
        if ($this->photo) {
            return $this->photo->getPhotoUrl();
        }
        return $this->getDummyImage();
    }

    public function getThumbUrl()
    {
        if ($this->photo) {
            return $this->photo->getThumbUrl();
        }
        return $this->getDummyImage();
    }

    public function getUrl()
    {
        return route('site.brands.show', $this);
    }

    public function getProductsUrl()
    {
        return route('site.products', ['filters' => "brand,{$this->id}-{$this->slug}"]);
    }

    public function hasLogo()
    {
        return $this->logo;
    }

    public function hasPhoto()
    {
        return $this->photo;
    }

    public function getCategory()
    {
        $product = $this->products()->first();

        return $product->category ?? null;
    }


}
