<?php

namespace App\Model;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Cache;
use Utils;
use Str;
use App\Feed\Attribute as FeedAttribute;

class Category extends Model
{

    protected $guarded = ['id'];

    /**
     * The attributes that should be casted to native types.
     *
     * @var array
     */
    protected $casts = [
        // 'options' => 'array', // JSON serialize
    ];

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
//    protected $fillable = [
//        'name', 'parent_id', 'priority', 'show_in_menu', 'is_active',
//    ];
    public static $tree = null;

    protected $disable_parent_checks = false;

    public static function boot()
    {
        parent::boot();

        static::saving(function ($model) {
            if ($model->parentCat) {
                $model->path = $model->parentCat->name . ' > ' . $model->name;
                if (!$model->disable_parent_checks) {
                    $model->parentCat->has_children = true;
                    $model->parentCat->save();
                }
//                $model->slug = Str::slug(implode(' ', [$model->parentCat->name, $model->name]));
            } else {
                $model->path = $model->name;
            }
            if ($model->can_create_articles || !$model->slug) {
                $model->slug = Str::slug($model->name);
            }
        });

        static::creating(function ($model) {
            $model->priority = 9999;
        });

        static::saved(function ($model) {
            if ($model->parent_id) {
                return;
            }
            // we simply save the children category to trigger saving event (so we can update path and slug!)
            $model->children->each(function ($item) {
                $item->disable_parent_checks = true;
                $item->save();
            });
        });

        static::deleting(function ($model) {
            $model->children->each(function ($item) {
                $item->delete();
            });
            // reset children flag if we delete all children from a parent categry.
            if ($model->parentCat && $model->parentCat->children->count() <= 1) {
                $model->parentCat->has_children = false;
                $model->parentCat->save();
            }
            // @todo: if no children in a parent category set has_children = false.
        });

        // we want the categories to be sorted always by priority.
//        static::addGlobalScope('order', function (Builder $builder) {
//            $builder->orderBy('priority', 'asc');
//        });
    }

    public function scopeRoot($query)
    {
        $query->where('parent_id', '=', 0);
    }

    public function scopeOrdered($query)
    {
        $query->orderBy('slug');
    }

    public function scopeActive($query)
    {
        $query->where('is_active', true);
    }

    public function scopePriority($query)
    {
        $query->orderBy('priority', 'asc');
    }


    public function scopeRecipe($query)
    {
        $query->where('parent_id', config('categories_mapping.recipes'));
    }

//    public function getRouteKeyName()
//    {
//        return 'slug';
//    }

    public function parentCat()
    {
        return $this->belongsTo(self::class, 'parent_id');

        // return $this->belongsTo(self::class, 'parent_id')->where('parent_id', 0);
    }

    public function children()
    {
        return $this->hasMany(self::class, 'parent_id');
    }


    public function subcategories()
    {
        return $this->children();
    }

    public function getAllGrouped()
    {
        $aux = [];
        $items = $this->root()->get();
        foreach ($items as $item) {
            $aux[$item->name] = $item->children->pluck('name', 'id')->toArray();
        }
        return $aux;
    }

    /**
     * Get products thru pivot table.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function articles()
    {
        return $this->hasMany('App\Model\Article');
    }


    public static function tree(&$categories = null, $parent_id = 0, $depth = 0)
    {
        // cached?
//        if (self::$tree) {
//            return self::$tree;
//        }
        $tree = collect();
        // feed initial categories
        if ($categories === null) {
            $categories = self::get();
        }
        // add to the tree recursively..
        foreach ($categories as $category) {
            if ($category->parent_id === $parent_id) {
                $category->depth = $depth;
                $category->subcats = self::tree($categories, $category->id, $depth + 1);
                $tree->push($category); // $category->toArray();
            }
        }
        // cache if origincal call
//        if ($is_initial_call) {
//            return self::$tree = $tree;
//        }
        return $tree;
    }


    public function getUrl($category = null)
    {
        if (in_array(config('categories_mapping.recipes'), [$this->id, $this->parent_id])) {
            return route('site.recipes.index', $this);
        }
        return route('site.articles.index', [$this, $category]);

        $category = array_search($this->id, config('categories_mapping'));
        if (!$category) {
            $category = array_search($this->parent_id, config('categories_mapping'));
        }

        return $category ? route("site.{$category}") : '#';
    }


    public function getPhotoUrl()
    {
        return Utils::storageUrl('categories', $this->photo);
    }

    public function isPage()
    {
        return $this->id == config('categories_mapping.pages');
    }

    public function isArticle()
    {
        return $this->id == config('categories_mapping.articles');
    }

}
