<?php

/**
 * Universal (or something like dat) payment dealer.
 *
 * (c) Sergiu Draga 12.2018
 * (c) Icetech 2018
 */

namespace Icetech\Payment\Controllers\Site;

use Artisan;
use App\Http\Controllers\Controller;
use App\Model\Order;
use Icetech\Payment\Models\Transaction;
use Payment;

class PaymentController extends Controller
{

    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            // we set the payment mode based on env.
            Payment::setTestMode(config('app.env') !== 'production');

            return $next($request);
        });
    }

    /**
     *  Generate the payment link, redirect to Payment processor
     */
    public function redirect()
    {
        if (!request('order_code')) {
            dd('We need the order code, rite?');
        }

        // find da order
        $order = Order::where('code', request('order_code'))->first();
        if (!$order) {
            dd('Nexam order');
        }

        // don't allow multiple payments.
        if ($order->is_paid) {
            dd('Already paid');
        }

        // you cannot purchase an order made in different envirenment.
        if ($order->payment_mode != (config('app.env') === 'production' ? 'live' : 'test')) {
            dd('Access denied');
        }


        // prepare urls to be sent to payment gateway.
        $params = Payment::setOrderId($order->code)
            ->setOrder($order)
            ->setAmount($order->getTotalPrice())
            ->setInstallment($order->hasInstallment(), $order->hasInstallment() ? $order->installment_period : null) // @quikfix
            ->setDetails($order->billingContact)
            ->generatePurchaseParams();;

        // some payment provides have directUrl, so we don't need a form mimic page.
        if (!empty($params['directUrl'])) {
            return redirect($params['directUrl']);
        }

        return view('payment::redirect', compact('params'));
    }

    /**
     *  After the client complete the payment process, he/she will be returned here.
     */
    public function status()
    {
        if (request('test2')) {
            $track_conversion = true;
//            dd('fa');
            Payment::setOrderId(request('orderId'));
            return Payment::callbackOutput($track_conversion);
        }
        $order = Order::where('code', request(Payment::getOrderIdParamName()))->first();

        if (!$order) {
            dd('Nexam order');
        }
        return view('payment::status', compact('order') + [
                'transaction' => $order->transaction,
            ]);
    }

    /**
     * This is the callback the payment processor  should call imediately or with a delay.
     * This will update our payment status.
     */
    public function confirm(Transaction $transaction)
    {
        $response = Payment::response()->getResponseData();

        // save order status
        $order = Order::where('code', Payment::getOrderId())->first();
        $track_conversion = false;
        if ($order) {
            $order->payment_status = $response->getAction();
            $order->payment_status_code = $response->getErrorCode();
            $order->is_paid = $response->getAction() === Payment::getPaidConst();
            $order->save();

            // if this is paid or preauth, annouce the user.
            if ($order->is_paid || $response->getAction() === Payment::getPreauthConst()) {
                $track_conversion = true;
                $order->sendEmailConfirmed();
            }
        }

        // save transaction
        $transaction->fill([
            'order_code' => Payment::getOrderId(),
            'has_error' => $response->hasError(),
            'status' => $response->getAction(),
            'status_code' => $response->getErrorCode(),
            'status_text' => $response->getMessage(),
            'response' => $response->getRaw(),
            'external_order_id' => $response->getExternalId(),
            'order_id' => $order ? $order->id : null,
        ])->save();


        return Payment::callbackOutput($track_conversion);
    }

    public function deposit()
    {
        $order = Order::where('code', request(Payment::getOrderIdParamName()))->first();

        if (!$order) {
            return 'Order not found';
        }

        if (empty($order->transaction->external_order_id)) {
            return 'No transaction for current order';
        }
        $response = Payment::setOrderId($order->transaction->external_order_id)
            ->setAmount($order->getTotalPrice())
            ->deposit();

        if ($response->status) {
            $order->payment_status = Payment::getPaidConst();
            $order->payment_status_code = $response->code;
            $order->is_paid = true;

            $order->save();

            // annouce the user.
            $order->sendEmailConfirmed();
        }


        $transaction = new Transaction;
        $transaction->fill([
            'order_code' => $order->code,
            'status_text' => 'deposit',
            'has_error' => !$response->status,
            'response' => $response->raw,
            'external_order_id' =>$order->transaction->external_order_id,
            'order_id' => $order->id,
        ])->save();

        if (auth()->check() && auth()->user()->isAdmin()) {
            return back();
        }

        return redirect(route('payment.status', [
            Payment::getOrderIdParamName() => $order->code,
        ]));

    }

    public function reverse()
    {
        $order = Order::where('code', request(Payment::getOrderIdParamName()))->first();

        if (!$order) {
            return 'Order not found';
        }

        if (empty($order->transaction->external_order_id)) {
            return 'No transaction for current order';
        }
        $response = Payment::setOrderId($order->transaction->external_order_id)
            ->setAmount($order->getTotalPrice())
            ->reverse();

        if ($response->status) {
            $order->payment_status = Payment::getCanceledConst();
//            $order->payment_status_code = $response->code;
            $order->is_paid = false;
            $order->save();
        }

        $transaction = new Transaction;
        $transaction->fill([
            'order_code' => $order->code,
            'status_text' => 'reverse',
            'has_error' => !$response->status,
            'response' => $response->raw,
            'external_order_id' =>$order->transaction->external_order_id,
            'order_id' => $order->id,
        ])->save();

        if (auth()->check() && auth()->user()->isAdmin()) {
            return back();
        }

        return redirect(route('payment.status', [
            Payment::getOrderIdParamName() => $order->code,
        ]));
    }

    public function migrate()
    {
        // 1. change env from production to dev
        // 2. change mysql user permissions if needed (ALTER + CREATE)
        Artisan::call('migrate', [
            '--force' => true,
            '--path' => realpath(__DIR__ . '/../' . '/migrations'),
        ]);
        return Artisan::output();
    }

}
